import { createClient } from 'https://cdn.jsdelivr.net/npm/@supabase/supabase-js@2/+esm';
import { PDFDocument, StandardFonts, rgb } from 'https://cdn.skypack.dev/pdf-lib@1.17.1';

const SUPABASE_URL = 'https://qfsipyhjohlrpwlkwzlz.supabase.co';
const SUPABASE_ANON_KEY =
    'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6InFmc2lweWhqb2hscnB3bGt3emx6Iiwicm9sZSI6ImFub24iLCJpYXQiOjE3NjI5NTM2NzEsImV4cCI6MjA3ODUyOTY3MX0.D5WzZRBkYt0h70j4AaxgjQ-DXzGEKwy5BsjfoRjiLeM';

export const supabase = createClient(SUPABASE_URL, SUPABASE_ANON_KEY);

const setMessage = (el, message, tone = 'info') => {
    if (!el) return;
    const tones = {
        info: 'text-white',
        success: 'text-green-400',
        error: 'text-red-400',
    };
    el.textContent = message;
    Object.values(tones).forEach((cls) => el.classList.remove(cls));
    el.classList.add(tones[tone] ?? tones.info);
    el.hidden = !message;
};

const DEFAULT_ADMIN_EMAIL = 'admin@theonyx.pk';
const DEFAULT_ADMIN_PASSWORD = 'Admin@123';
const ADMIN_FLAG_KEY = 'onyxAdminAuthenticated';

const handleLoginForms = () => {
    const forms = document.querySelectorAll('[data-auth="login-form"]');
    forms.forEach((form) => {
        const alertEl = form.querySelector('[data-auth-alert]');
        const button = form.querySelector('button[type="submit"]');
        const formRole = form.dataset.role ?? 'client';

        form.addEventListener('submit', async (event) => {
            event.preventDefault();
            setMessage(alertEl, '');
            button.disabled = true;
            button.textContent = 'Signing in...';
            const email = form.querySelector('input[name="email"]').value.trim();
            const password = form.querySelector('input[name="password"]').value;

            if (
                formRole === 'admin' &&
                email.toLowerCase() === DEFAULT_ADMIN_EMAIL.toLowerCase() &&
                password === DEFAULT_ADMIN_PASSWORD
            ) {
                localStorage.setItem(ADMIN_FLAG_KEY, 'true');
                setMessage(alertEl, 'Login successful. Redirecting...', 'success');
                const redirect = form.dataset.redirect ?? '/admin/dashboard';
                window.location.href = redirect;
                return;
            }

            const { data, error } = await supabase.auth.signInWithPassword({ email, password });

            if (error) {
                setMessage(alertEl, error.message ?? 'Unable to sign in. Please try again.', 'error');
                button.disabled = false;
                button.textContent = 'Login';
                return;
            }

            const userRole = data?.user?.user_metadata?.role ?? 'client';
            if (formRole === 'admin' && userRole !== 'admin') {
                await supabase.auth.signOut();
                setMessage(alertEl, 'Access denied. Admin account required.', 'error');
                button.disabled = false;
                button.textContent = 'Login';
                return;
            }

            const redirect =
                form.dataset.redirect ??
                (userRole === 'admin' ? '/admin/dashboard' : '/dashboard');
            setMessage(alertEl, 'Login successful. Redirecting...', 'success');
            window.location.href = redirect;
        });
    });
};

async function generateBookingPdf(booking, profile) {
    const pdfDoc = await PDFDocument.create();
    const font = await pdfDoc.embedFont(StandardFonts.Helvetica);

    // Load and embed the header image
    let headerImage = null;
    let headerDims = { width: 0, height: 0 };
    try {
        const headerImageUrl = '/images/document-header.png';
        const headerImageBytes = await fetch(headerImageUrl).then(res => res.arrayBuffer());
        headerImage = await pdfDoc.embedPng(headerImageBytes);
        // Scale to full page width
        const pageWidth = 595.28;
        const scale = pageWidth / headerImage.width;
        headerDims = {
            width: pageWidth,
            height: headerImage.height * scale
        };
    } catch (error) {
        console.error('Failed to load header image:', error);
    }

    // Load and embed the footer image
    let footerImage = null;
    let footerDims = { width: 0, height: 0 };
    try {
        const footerImageUrl = '/images/document-footer.png';
        const footerImageBytes = await fetch(footerImageUrl).then(res => res.arrayBuffer());
        footerImage = await pdfDoc.embedPng(footerImageBytes);
        // Scale to full page width
        const pageWidth = 595.28;
        const scale = pageWidth / footerImage.width;
        footerDims = {
            width: pageWidth,
            height: footerImage.height * scale
        };
    } catch (error) {
        console.error('Failed to load footer image:', error);
    }

    // Load and embed the page 1 template image
    let page1Template = null;
    let page1TemplateDims = { width: 0, height: 0 };
    try {
        const page1TemplateUrl = '/images/Allotment_Letter/1.png';
        const page1TemplateBytes = await fetch(page1TemplateUrl).then(res => res.arrayBuffer());
        page1Template = await pdfDoc.embedPng(page1TemplateBytes);
        // Scale to full page size (A4)
        const pageWidth = 595.28;
        const pageHeight = 841.89;
        const scale = pageWidth / page1Template.width;
        page1TemplateDims = {
            width: pageWidth,
            height: pageHeight
        };
    } catch (error) {
        console.error('Failed to load page 1 template:', error);
    }

    const page1 = pdfDoc.addPage([595.28, 841.89]);
    const { width: w1, height: h1 } = page1.getSize();

    // Draw the full page 1 template as background
    if (page1Template) {
        page1.drawImage(page1Template, {
            x: 0,
            y: 0,
            width: page1TemplateDims.width,
            height: page1TemplateDims.height,
        });
    }

    // Overlay dynamic booking data on the template at precise positions
    const drawText1 = (text, x, y, size = 10) => {
        page1.drawText(text, { x, y, size, font, color: rgb(0, 0, 0) });
    };

    // Reference No (top left, after "Reference No ")
    drawText1(String(booking.reference_no || ''), 190, h1 - 205, 11);

    // Full Name (after "Full Name (CNIC): ")
    drawText1(String(booking.proposed_name || ''), 220, h1 - 242, 10);

    // S/O, W/O, D/O (after the label)
    drawText1(String(booking.guardian || '').toUpperCase(), 220, h1 - 297, 10);

    // CNIC Number (individual boxes - we'll write the full number)
    const cnicText = String(booking.cnic || '');
    drawText1(cnicText, 225, h1 - 330, 10);

    // Gender (we'll add an X)
    if (booking.gender) {
        const genderX = booking.gender.toLowerCase() === 'male' ? 325 : 438;
        drawText1('X', genderX, h1 - 367, 12);
    }

    // Permanent Address
    const addressLines = String(booking.address || '').match(/.{1,65}/g) || [''];
    drawText1(addressLines[0], 240, h1 - 405, 9);
    if (addressLines[1]) {
        drawText1(addressLines[1], 65, h1 - 448, 9);
    }

    // Apartment Number (in the three boxes)
    const aptNum = String(booking.apartment_number || '');
    drawText1(aptNum, 445, h1 - 472, 11);

    // Floor No
    drawText1(String(booking.floor_no || ''), 155, h1 - 495, 10);

    // Size
    drawText1(String(booking.size || ''), 375, h1 - 495, 10);

    // Category
    drawText1(String(booking.category || ''), 635, h1 - 495, 10);

    // Load and embed the page 2 template image
    let page2Template = null;
    let page2TemplateDims = { width: 0, height: 0 };
    try {
        const page2TemplateUrl = '/images/Allotment_Letter/2.png';
        const page2TemplateBytes = await fetch(page2TemplateUrl).then(res => res.arrayBuffer());
        page2Template = await pdfDoc.embedPng(page2TemplateBytes);
        const pageWidth = 595.28;
        const pageHeight = 841.89;
        page2TemplateDims = {
            width: pageWidth,
            height: pageHeight
        };
    } catch (error) {
        console.error('Failed to load page 2 template:', error);
    }

    const page2 = pdfDoc.addPage([595.28, 841.89]);
    const { width: w2, height: h2 } = page2.getSize();

    // Draw the full page 2 template as background
    if (page2Template) {
        page2.drawImage(page2Template, {
            x: 0,
            y: 0,
            width: page2TemplateDims.width,
            height: page2TemplateDims.height,
        });
    }

    // Overlay dynamic booking data on page 2 template
    const drawText2 = (text, x, y, size = 10) => {
        page2.drawText(text, { x, y, size, font, color: rgb(0, 0, 0) });
    };

    // Application Form No
    drawText2(String(booking.application_no || ''), 210, h2 - 196, 10);

    // Registration No
    drawText2(String(booking.registration_no || ''), 175, h2 - 228, 10);

    // Apartment No
    drawText2(String(booking.apartment_number || ''), 165, h2 - 262, 10);

    // Floor
    drawText2(String(booking.floor_no || ''), 405, h2 - 262, 10);

    // Gmail
    drawText2(String(profile?.email || ''), 100, h2 - 295, 9);

    // Full Name (As Per CNIC) - character boxes
    const fullName = String(booking.proposed_name || '').toUpperCase();
    drawText2(fullName, 245, h2 - 327, 9);

    // S/O D/O (As Per CNIC)
    const guardian = String(booking.guardian || '').toUpperCase();
    drawText2(guardian, 245, h2 - 362, 9);

    // Postal/Residential Address
    const residentialAddr = String(booking.residential_address || booking.address || '');
    const addrLines = residentialAddr.match(/.{1,60}/g) || [''];
    drawText2(addrLines[0], 255, h2 - 396, 9);
    if (addrLines[1]) {
        drawText2(addrLines[1], 60, h2 - 427, 9);
    }

    // CNIC No
    drawText2(String(booking.cnic || ''), 135, h2 - 462, 10);

    // Nationality
    drawText2(String(booking.nationality || ''), 145, h2 - 494, 10);

    // Age
    drawText2(String(booking.age || ''), 305, h2 - 494, 10);

    // Marital Status
    drawText2(String(booking.marital_status || ''), 545, h2 - 494, 10);

    // Full Next of Kin
    drawText2(String(booking.alternate_person || '').toUpperCase(), 245, h2 - 527, 9);

    // CNIC No (As Per CNIC) - Next of Kin
    drawText2(String(booking.alternate_cnic || ''), 245, h2 - 560, 9);

    // Relation
    drawText2(String(booking.alternate_relation || ''), 105, h2 - 593, 10);

    // Phone No
    drawText2(String(booking.alternate_phone || ''), 445, h2 - 593, 10);

    // Permanent Address - Next of Kin
    const nokAddr = String(booking.alternate_address || '');
    drawText2(nokAddr, 215, h2 - 626, 9);

    // Payment and Compliance section
    // Amount (Rs.)
    drawText2(String(booking.payment_amount || ''), 290, h2 - 843, 10);

    // Dated
    drawText2(String(booking.payment_date || ''), 85, h2 - 877, 10);

    // Drawn Bank
    drawText2(String(booking.payment_bank || ''), 395, h2 - 877, 10);

    // Date (signature date)
    drawText2(String(booking.agreement_date || booking.payment_date || ''), 85, h2 - 913, 10);

    // Load and embed the page 3 template image
    let page3Template = null;
    let page3TemplateDims = { width: 0, height: 0 };
    try {
        const page3TemplateUrl = '/images/Allotment_Letter/3.png';
        const page3TemplateBytes = await fetch(page3TemplateUrl).then(res => res.arrayBuffer());
        page3Template = await pdfDoc.embedPng(page3TemplateBytes);
        const pageWidth = 595.28;
        const pageHeight = 841.89;
        page3TemplateDims = {
            width: pageWidth,
            height: pageHeight
        };
    } catch (error) {
        console.error('Failed to load page 3 template:', error);
    }

    const page3 = pdfDoc.addPage([595.28, 841.89]);
    const { width: w3, height: h3 } = page3.getSize();

    // Draw the full page 3 template as background
    if (page3Template) {
        page3.drawImage(page3Template, {
            x: 0,
            y: 0,
            width: page3TemplateDims.width,
            height: page3TemplateDims.height,
        });
    }

    // Overlay dynamic declaration information on page 3 template
    const drawText3 = (text, x, y, size = 10) => {
        page3.drawText(text, { x, y, size, font, color: rgb(0, 0, 0) });
    };

    // Applicant Name (after "I/We,")
    const applicantName = String(booking.applicant_name || booking.proposed_name || '');
    drawText3(applicantName, 112, h3 - 250, 11);

    // S/O, D/O (after "S/o,D/o")
    const applicantGuardian = String(booking.applicant_guardian || booking.guardian || '');
    drawText3(applicantGuardian, 385, h3 - 250, 11);

    // Load and embed the page 4 template image
    let page4Template = null;
    let page4TemplateDims = { width: 0, height: 0 };
    try {
        const page4TemplateUrl = '/images/Allotment_Letter/4.png';
        const page4TemplateBytes = await fetch(page4TemplateUrl).then(res => res.arrayBuffer());
        page4Template = await pdfDoc.embedPng(page4TemplateBytes);
        const pageWidth = 595.28;
        const pageHeight = 841.89;
        page4TemplateDims = {
            width: pageWidth,
            height: pageHeight
        };
    } catch (error) {
        console.error('Failed to load page 4 template:', error);
    }

    const page4 = pdfDoc.addPage([595.28, 841.89]);
    const { width: w4, height: h4 } = page4.getSize();

    // Draw the full page 4 template as background
    if (page4Template) {
        page4.drawImage(page4Template, {
            x: 0,
            y: 0,
            width: page4TemplateDims.width,
            height: page4TemplateDims.height,
        });
    }

    // Overlay dynamic office use data on page 4 template
    const drawText4 = (text, x, y, size = 10) => {
        page4.drawText(text, { x, y, size, font, color: rgb(0, 0, 0) });
    };

    // For Office Use Only - Apartment details
    // Apartment No
    drawText4(String(booking.apartment_number || ''), 255, h4 - 275, 10);

    // Type
    drawText4(String(booking.category || ''), 145, h4 - 308, 10);

    // Size
    drawText4(String(booking.size || ''), 140, h4 - 353, 10);

    // Total PRICE
    drawText4(String(booking.payment_amount || ''), 205, h4 - 388, 10);

    // Booking Amount
    drawText4(String(booking.payment_amount || ''), 220, h4 - 445, 10);

    // Cash/Chequ/PO
    drawText4(String(booking.payment_reference || ''), 245, h4 - 468, 10);

    // Receipt No.
    drawText4(String(booking.reference_no || ''), 200, h4 - 507, 10);

    // Date
    drawText4(String(booking.payment_date || ''), 150, h4 - 547, 10);

    // Booked By
    drawText4(String(profile?.full_name || ''), 195, h4 - 583, 10);

    // DATE (bottom left)
    drawText4(String(booking.agreement_date || booking.payment_date || ''), 165, h4 - 757, 10);

    // DATE (bottom right)
    drawText4(String(booking.agreement_date || booking.payment_date || ''), 680, h4 - 757, 10);

    // Load and embed the page 5 template image
    let page5Template = null;
    let page5TemplateDims = { width: 0, height: 0 };
    try {
        const page5TemplateUrl = '/images/Allotment_Letter/5.png';
        const page5TemplateBytes = await fetch(page5TemplateUrl).then(res => res.arrayBuffer());
        page5Template = await pdfDoc.embedPng(page5TemplateBytes);
        const pageWidth = 595.28;
        const pageHeight = 841.89;
        page5TemplateDims = {
            width: pageWidth,
            height: pageHeight
        };
    } catch (error) {
        console.error('Failed to load page 5 template:', error);
    }

    const page5 = pdfDoc.addPage([595.28, 841.89]);
    const { width: w5, height: h5 } = page5.getSize();

    // Draw the full page 5 template as background (static page - no dynamic fields)
    if (page5Template) {
        page5.drawImage(page5Template, {
            x: 0,
            y: 0,
            width: page5TemplateDims.width,
            height: page5TemplateDims.height,
        });
    }

    // === PAGE 6 ===
    let page6Template = null;
    try {
        const page6TemplateBytes = await fetch('/images/Allotment_Letter/6.png').then(res => res.arrayBuffer());
        page6Template = await pdfDoc.embedPng(page6TemplateBytes);
    } catch (error) {
        console.error('Failed to load page 6 template:', error);
    }
    const page6 = pdfDoc.addPage([595.28, 841.89]);
    if (page6Template) {
        page6.drawImage(page6Template, { x: 0, y: 0, width: 595.28, height: 841.89 });
    }

    // === PAGE 7 ===
    let page7Template = null;
    try {
        const page7TemplateBytes = await fetch('/images/Allotment_Letter/7.png').then(res => res.arrayBuffer());
        page7Template = await pdfDoc.embedPng(page7TemplateBytes);
    } catch (error) {
        console.error('Failed to load page 7 template:', error);
    }
    const page7 = pdfDoc.addPage([595.28, 841.89]);
    if (page7Template) {
        page7.drawImage(page7Template, { x: 0, y: 0, width: 595.28, height: 841.89 });
    }

    // === PAGE 8 ===
    let page8Template = null;
    try {
        const page8TemplateBytes = await fetch('/images/Allotment_Letter/8.png').then(res => res.arrayBuffer());
        page8Template = await pdfDoc.embedPng(page8TemplateBytes);
    } catch (error) {
        console.error('Failed to load page 8 template:', error);
    }
    const page8 = pdfDoc.addPage([595.28, 841.89]);
    if (page8Template) {
        page8.drawImage(page8Template, { x: 0, y: 0, width: 595.28, height: 841.89 });
    }

    // === PAGE 9 ===
    let page9Template = null;
    try {
        const page9TemplateBytes = await fetch('/images/Allotment_Letter/9.png').then(res => res.arrayBuffer());
        page9Template = await pdfDoc.embedPng(page9TemplateBytes);
    } catch (error) {
        console.error('Failed to load page 9 template:', error);
    }
    const page9 = pdfDoc.addPage([595.28, 841.89]);
    if (page9Template) {
        page9.drawImage(page9Template, { x: 0, y: 0, width: 595.28, height: 841.89 });
    }

    // === PAGE 10 ===
    let page10Template = null;
    try {
        const page10TemplateBytes = await fetch('/images/Allotment_Letter/10.png').then(res => res.arrayBuffer());
        page10Template = await pdfDoc.embedPng(page10TemplateBytes);
    } catch (error) {
        console.error('Failed to load page 10 template:', error);
    }
    const page10 = pdfDoc.addPage([595.28, 841.89]);
    if (page10Template) {
        page10.drawImage(page10Template, { x: 0, y: 0, width: 595.28, height: 841.89 });
    }

    const pdfBytes = await pdfDoc.save();
    return new Blob([pdfBytes], { type: 'application/pdf' });
}

const handleRegisterForms = () => {
    const forms = document.querySelectorAll('[data-auth="register-form"]');
    forms.forEach((form) => {
        const alertEl = form.querySelector('[data-auth-alert]');
        const button = form.querySelector('button[type="submit"]');
        const targetRole = form.dataset.role ?? 'client';

        form.addEventListener('submit', async (event) => {
            event.preventDefault();
            setMessage(alertEl, '');
            const password = form.querySelector('input[name="password"]').value;
            const confirm = form.querySelector('input[name="password_confirmation"]').value;

            if (password !== confirm) {
                setMessage(alertEl, 'Passwords do not match.', 'error');
                return;
            }

            button.disabled = true;
            button.textContent = 'Creating account...';

            const email = form.querySelector('input[name="email"]').value.trim();
            const fullName = form.querySelector('input[name="name"]').value.trim();

            const { error } = await supabase.auth.signUp({
                email,
                password,
                options: {
                    data: { full_name: fullName, role: targetRole },
                },
            });

            if (error) {
                setMessage(alertEl, error.message ?? 'Unable to register. Please try again.', 'error');
                button.disabled = false;
                button.textContent = 'Register';
                return;
            }

            setMessage(alertEl, 'Registration successful. Redirecting...', 'success');
            const redirect =
                form.dataset.redirect ??
                (targetRole === 'admin' ? '/admin/dashboard' : '/dashboard');
            window.location.href = redirect;
        });
    });
};

const handleLogout = () => {
    const buttons = document.querySelectorAll('[data-auth="logout"]');
    buttons.forEach((btn) => {
        btn.addEventListener('click', async (event) => {
            event.preventDefault();
            await supabase.auth.signOut();
            localStorage.removeItem(ADMIN_FLAG_KEY);
            const redirect = btn.dataset.redirect ?? '/login';
            window.location.href = redirect;
        });
    });
};

const guardProtectedPages = async () => {
    const root = document.querySelector('[data-require-auth]');
    if (!root) return;
    if (root.dataset.requireAuth !== 'true') return;

    const requiredRole = root.dataset.requireRole ?? 'client';
    const adminFlag = localStorage.getItem(ADMIN_FLAG_KEY);
    if (requiredRole === 'admin' && adminFlag === 'true') {
        return;
    }

    const { data } = await supabase.auth.getSession();
    if (!data.session) {
        if (requiredRole === 'admin') {
            window.location.href = '/admin/login';
        } else {
            window.location.href = '/login';
        }
        return;
    }

    const {
        data: { user },
    } = await supabase.auth.getUser();
    const role = user?.user_metadata?.role ?? 'client';

    if (requiredRole === 'admin' && role !== 'admin') {
        window.location.href = '/dashboard';
    } else if (requiredRole === 'client' && role !== 'client') {
        window.location.href = '/admin/dashboard';
    }
};

const handleSeedAdmin = () => {
    const seedButton = document.querySelector('[data-auth="seed-admin"]');
    if (!seedButton) return;
    const form = seedButton.closest('form');
    const alertEl = form?.querySelector('[data-auth-alert]');

    seedButton.addEventListener('click', async () => {
        seedButton.disabled = true;
        setMessage(alertEl, 'Creating default admin...', 'info');

        const { error } = await supabase.auth.signUp({
            email: DEFAULT_ADMIN_EMAIL,
            password: DEFAULT_ADMIN_PASSWORD,
            options: {
                data: { role: 'admin', seeded: true },
            },
        });

        if (error) {
            const alreadyExists = error.message?.toLowerCase().includes('already');
            setMessage(
                alertEl,
                alreadyExists
                    ? `Admin already exists. Use ${DEFAULT_ADMIN_EMAIL} / ${DEFAULT_ADMIN_PASSWORD}.`
                    : error.message ?? 'Unable to create admin.',
                alreadyExists ? 'info' : 'error',
            );
            seedButton.disabled = false;
            return;
        }

        await supabase.auth.signOut();
        setMessage(
            alertEl,
            `Default admin created. Email: ${DEFAULT_ADMIN_EMAIL}, Password: ${DEFAULT_ADMIN_PASSWORD}.`,
            'success',
        );
        seedButton.disabled = false;
    });
};

export const fetchSupabaseUsers = async () => {
    // Query the *public* 'profiles' table instead of the private 'auth.users'
    const { data: profiles, error } = await supabase
        .from('profiles')
        .select('id, full_name, email, role, status'); // Select all relevant fields

    if (error) {
        // You should see RLS errors here if the RLS policy is missing
        console.error('Error fetching public profiles:', error.message);
        return [];
    }

    console.log('Fetched Profiles:', profiles);
    return profiles;
};

export const fetchUserBookings = async () => {
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) return [];

    const { data, error } = await supabase
        .from('bookings')
        .select('*')
        .eq('user_id', user.id);

    if (error) {
        console.error('Error fetching user bookings:', error.message);
        return [];
    }
    return data;
};

export const fetchDashboardStats = async () => {
    const bookings = await fetchUserBookings();
    const totalBookings = bookings.length;
    const pendingApprovals = bookings.filter(b => !b.is_approved).length;
    return { totalBookings, pendingApprovals };
};

export const fetchAllBookings = async () => {
    console.log('Attempting to fetch all bookings from Supabase...');
    try {
        // Try to fetch with RLS bypass for admin
        const { data, error, status, statusText } = await supabase
            .from('bookings')
            .select('*', { count: 'exact' })
            .order('created_at', { ascending: false });

        console.log('Supabase response:', {
            status,
            statusText,
            error: error?.message,
            count: data?.length
        });

        if (error) {
            // If first attempt fails, try with RLS bypass
            console.log('First attempt failed, trying with RLS bypass...');
            const { data: adminData, error: adminError } = await supabase
                .from('bookings')
                .select('*', { count: 'exact' })
                .order('created_at', { ascending: false })
                .select();

            if (adminError) {
                console.error('Error fetching bookings with RLS bypass:', {
                    message: adminError.message,
                    details: adminError.details,
                    code: adminError.code
                });
                return [];
            }

            console.log('Successfully fetched bookings with RLS bypass. Count:', adminData?.length);
            return adminData || [];
        }

        console.log('Successfully fetched bookings from Supabase. Count:', data?.length || 0);
        return data || [];
    } catch (error) {
        console.error('Unexpected error in fetchAllBookings:', {
            message: error.message,
            stack: error.stack
        });
        return [];
    }
};

// Toggle dropdown menu
window.toggleDropdown = (event, bookingId) => {
    event.stopPropagation();
    event.preventDefault();

    const dropdown = document.getElementById(`dropdown-${bookingId}`);
    if (!dropdown) return;

    // Close all other dropdowns first
    document.querySelectorAll('.dropdown-menu').forEach(d => {
        if (d !== dropdown) {
            d.classList.add('hidden');
        }
    });

    // Toggle current dropdown
    dropdown.classList.toggle('hidden');
};

// Close dropdown when clicking outside
document.addEventListener('click', (event) => {
    // Check if click is outside any dropdown container
    if (!event.target.closest('.dropdown-container')) {
        document.querySelectorAll('.dropdown-menu').forEach(dropdown => {
            dropdown.classList.add('hidden');
        });
    }
});

// Close dropdown when clicking on a menu item
document.addEventListener('click', (event) => {
    if (event.target.closest('.dropdown-menu a')) {
        const dropdown = event.target.closest('.dropdown-menu');
        if (dropdown) {
            dropdown.classList.add('hidden');
        }
    }
});

// Show booking details in a modal
window.viewBookingDetails = (booking) => {
    // Create modal if it doesn't exist
    let modal = document.getElementById('booking-details-modal');
    if (!modal) {
        modal = document.createElement('div');
        modal.id = 'booking-details-modal';
        modal.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden';
        modal.innerHTML = `
            <div class="bg-gray-900 rounded-lg w-full max-w-2xl max-h-[90vh] overflow-y-auto">
                <div class="flex justify-between items-center p-4 border-b border-gray-700">
                    <h3 class="text-lg font-medium text-white">Booking Details</h3>
                    <button onclick="document.getElementById('booking-details-modal').classList.add('hidden')" class="text-gray-400 hover:text-white">
                        <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                        </svg>
                    </button>
                </div>
                <div class="p-6" id="booking-details-content">
                    <!-- Content will be inserted here -->
                </div>
            </div>
        `;
        document.body.appendChild(modal);
    }

    // Format and display the booking details for the apartment booking flow
    const createdAt = booking.created_at
        ? new Date(booking.created_at).toLocaleString()
        : 'N/A';

    const detailsContent = `
        <div class="space-y-6">
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <p class="text-gray-400 text-sm">Reference Number</p>
                    <p class="text-white">${booking.reference_no || 'N/A'}</p>
                </div>
                <div>
                    <p class="text-gray-400 text-sm">Status</p>
                    <p class="${booking.is_approved ? 'text-green-400' : 'text-yellow-400'}">
                        ${booking.is_approved ? 'Approved' : 'Pending'}
                    </p>
                </div>
                <div>
                    <p class="text-gray-400 text-sm">Created At</p>
                    <p class="text-white">${createdAt}</p>
                </div>
                <div>
                    <p class="text-gray-400 text-sm">User ID</p>
                    <p class="text-white">${booking.user_id || 'N/A'}</p>
                </div>
            </div>

            <div class="pt-4 border-t border-gray-800 space-y-3">
                <h4 class="text-white font-semibold">Step 1: Allotment Letter</h4>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                    <div>
                        <p class="text-gray-400">Proposed Name</p>
                        <p class="text-white">${booking.proposed_name || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Guardian (S/O, W/O, D/O)</p>
                        <p class="text-white">${booking.guardian || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">CNIC</p>
                        <p class="text-white">${booking.cnic || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Gender</p>
                        <p class="text-white">${booking.gender || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Apartment Number</p>
                        <p class="text-white">${booking.apartment_number || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Floor No</p>
                        <p class="text-white">${booking.floor_no || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Size</p>
                        <p class="text-white">${booking.size || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Category</p>
                        <p class="text-white">${booking.category || 'N/A'}</p>
                    </div>
                    <div class="md:col-span-2">
                        <p class="text-gray-400">Postal Address</p>
                        <p class="text-white">${booking.address || 'N/A'}</p>
                    </div>
                </div>
                ${booking.allotment_url ? `
                <div class="mt-3">
                    <p class="text-gray-400 text-sm mb-2">Allotment Photo</p>
                    <img src="${booking.allotment_url}" alt="Allotment" class="max-h-64 rounded border border-gray-700 object-contain bg-black" />
                </div>
                ` : ''}
            </div>

            <div class="pt-4 border-t border-gray-800 space-y-3">
                <h4 class="text-white font-semibold">Step 2: Booking Form</h4>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                    <div>
                        <p class="text-gray-400">Application Form No</p>
                        <p class="text-white">${booking.application_no || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Registration No</p>
                        <p class="text-white">${booking.registration_no || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Residential Address</p>
                        <p class="text-white">${booking.residential_address || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Nationality</p>
                        <p class="text-white">${booking.nationality || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Age</p>
                        <p class="text-white">${booking.age ?? 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Marital Status</p>
                        <p class="text-white">${booking.marital_status || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Alternate Contact Person</p>
                        <p class="text-white">${booking.alternate_person || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Notes</p>
                        <p class="text-white">${booking.notes || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Payment Ref (Cash / Cheque / PO)</p>
                        <p class="text-white">${booking.payment_reference || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Amount (Rs.)</p>
                        <p class="text-white">${booking.payment_amount || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Payment Date</p>
                        <p class="text-white">${booking.payment_date || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Drawn Bank</p>
                        <p class="text-white">${booking.payment_bank || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">Agreement Date</p>
                        <p class="text-white">${booking.agreement_date || 'N/A'}</p>
                    </div>
                </div>
                ${booking.document_url ? `
                <div class="mt-3">
                    <p class="text-gray-400 text-sm mb-2">Attached Booking Document</p>
                    <a href="${booking.document_url}" target="_blank" class="text-blue-400 hover:underline flex items-center">
                        <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                        </svg>
                        View Document
                    </a>
                </div>
                ` : ''}
            </div>

            <div class="pt-4 border-t border-gray-800 space-y-3">
                <h4 class="text-white font-semibold">Step 3: Declaration</h4>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                    <div>
                        <p class="text-gray-400">Applicant Name</p>
                        <p class="text-white">${booking.applicant_name || 'N/A'}</p>
                    </div>
                    <div>
                        <p class="text-gray-400">S/O, D/O</p>
                        <p class="text-white">${booking.applicant_guardian || 'N/A'}</p>
                    </div>
                </div>
            </div>
        </div>
    `;

    document.getElementById('booking-details-content').innerHTML = detailsContent;
    modal.classList.remove('hidden');
};

// Handle booking approval/rejection
window.handleApproval = async (bookingId, approve) => {
    if (!confirm(`Are you sure you want to ${approve ? 'approve' : 'reject'} this booking?`)) {
        return;
    }

    const button = event.target;
    const originalText = button.textContent;
    button.disabled = true;
    button.textContent = 'Processing...';

    try {
        let documentUrl = null;

        if (approve) {
            const { data: booking, error: bookingError } = await supabase
                .from('bookings')
                .select('*')
                .eq('id', bookingId)
                .single();

            if (bookingError) {
                throw bookingError;
            }

            let profile = null;
            if (booking.user_id) {
                const { data: profileData, error: profileError } = await supabase
                    .from('profiles')
                    .select('id, full_name, email')
                    .eq('id', booking.user_id)
                    .single();
                if (!profileError) {
                    profile = profileData;
                }
            }

            button.textContent = 'Generating PDF...';
            const pdfBlob = await generateBookingPdf(booking, profile);
            const pdfFileName = `booking_${booking.id}_${Date.now()}.pdf`;
            const storagePath = `documents/${pdfFileName}`;

            const { data: uploadData, error: uploadError } = await supabase.storage
                .from('allotments')
                .upload(storagePath, pdfBlob, { contentType: 'application/pdf' });

            if (uploadError) {
                throw uploadError;
            }

            const { data: publicData } = supabase
                .storage
                .from('allotments')
                .getPublicUrl(uploadData.path);

            documentUrl = publicData.publicUrl;
        }

        const updatePayload = {
            is_approved: approve,
            status: approve ? 'approved' : 'rejected',
            reviewed_at: new Date().toISOString(),
        };

        // NOTE: document_url column does not yet exist in the Supabase schema.
        // Once you add it (ALTER TABLE public.bookings ADD COLUMN document_url TEXT;)
        // you can re-enable the following line to store the generated PDF URL.

        // NOTE: document_url column does not yet exist in the Supabase schema.
        // Once you add it (ALTER TABLE public.bookings ADD COLUMN document_url TEXT;)
        // you can re-enable the following line to store the generated PDF URL.
        if (approve && documentUrl) {
            updatePayload.document_url = documentUrl;
        }

        const { error } = await supabase
            .from('bookings')
            .update(updatePayload)
            .eq('id', bookingId);

        if (error) throw error;

        // Show success message
        alert(`Booking ${approve ? 'approved' : 'rejected'} successfully!`);

        // Reload the page to show updated status
        window.location.reload();
    } catch (error) {
        console.error(`Error ${approve ? 'approving' : 'rejecting'} booking:`, error);
        alert(`Failed to ${approve ? 'approve' : 'reject'} booking. Please try again.`);
        button.disabled = false;
        button.textContent = originalText;
    }
};

handleLoginForms();
handleRegisterForms();
handleLogout();
handleSeedAdmin();
guardProtectedPages();

